/**
 * 使い方：
 *  　const meter = new SGTMeter();
 *    meter.sens = 4;               //感度を設定
 *  　meter.position = TAポジション; //基準TA値を設定
 *    // PCのTA値またはPCの抵抗値を設定（両方設定する必要はなく、どちらかを設定すれば良い)
 *  　meter.ta = PCのTA値;           //PCのTA値を設定
 *  　meter.r = PCの抵抗値;           //PCの抵抗値を設定
 *  　let meterValue = meter.value;  //メーターの値を取得
 * 
 *    メーターの値は 左端：0、セット位置：15、目盛最大値：40、右端：42
 * 　　PCのTA値が表示範囲を超えている場合 0以下または42以上となる。
 */
class SGTMeter {
    #TA_MAX = 6.5;         //TAの最大値
    #TA_MIN = 0.9412;      //TAの最小値
    #TA_BASE = 2.0;        //感度補正の倍率を1とするTA値
    #TA_LOW_TH = 2.6;      //感度補正を穏やかにする閾値
    #TA_HIGH_TH = 4.3;     //感度補正を穏やかにする閾値
    #TA_LOW_RATIO = 0.1;   //感度補正を穏やかにする割合
    #TA_HIGH_RATIO = 0.6;  //感度補正を穏やかにする割合
    #R_CONST = 21250.0;    //TA計算に使用する抵抗値
    #R_TA20 = 5000.0;      //TAが2.0となる抵抗値
    #R_SENS_BASE = 3125.0; //感度を計算するための抵抗値
    #R_FALL = 860.0;       //TA2.5、感度4(補正前)でFALL目盛までリードする時の抵抗の減少量
    #SCALE_MAX = 40.0;     //メーター目盛の最大値
    #SCALE_SET = 15.0;     //セット位置のメーター目盛
    #SCALE_SIZE = 42;      //メーターのサイズ(最大値を超えて約2メモリ分の余白がある)

    #taDiffFallBase; //TAがTA_BASEのときに抵抗値が R_FALL 減少した場合のTAの変化量
    #taScaleCoef;    //1TAあたりの目盛の変化量

    constructor() {
        this.#taDiffFallBase = this.#taDiffFall(this.#TA_BASE);
    }

    // メーターのサイズ
    get scaleSize() {
        return this.#SCALE_SIZE;
    }

    // 基準TA
    #taPosition = this.#TA_BASE;
    get position() {
        return this.#taPosition;
    }
    set position(p) {
        this.#taPosition = p;
        this.#calcScaleCoef();
    }

    // PCのTA
    #ta = this.#TA_BASE;
    get ta() {
        return this.#ta;
    }
    set ta(pcTA) {
        this.#ta = pcTA;
        this.#r = this.#ta2r(pcTA);
    }

    // PCの抵抗値
    #r = this.#R_TA20;
    get r() {
        return this.#r;
    }
    set r(pcR) {
        this.#r = pcR;
        this.#ta = this.#r2ta(pcR);
    }

    // 感度
    #sens = 1.0;
    get sens() {
        return this.#sens;
    }
    set sens(s) {
        this.#sens = s;
        this.#calcScaleCoef();
    }

    // メーターの値
    get value() {
        return (this.#taPosition - this.#ta) * this.#taScaleCoef + this.#SCALE_SET;
    }

    //---------------------------------------------------------------------------------------------//

    /** TAから抵抗値を計算する */
    #ta2r(t) {
        return this.#R_CONST * (t - this.#TA_MIN) / (this.#TA_MAX - t);
    }
    /** 抵抗値からTAを計算する */
    #r2ta(r) {
        return r / (r + this.#R_CONST) * (this.#TA_MAX - this.#TA_MIN) + this.#TA_MIN;
    }
    /**
     * 引数のTA値にて抵抗値が R_FALL 減少した際のTAの変化量を計算する。
     * TAがTA_LOW_THからTA_HIGH_THを外れる場合は、感度補正を穏やかにするためにTA値を変換してから計算する。
     */
    #taDiffFall(t) {
        if (t < this.#TA_LOW_TH) {
            t = this.#TA_LOW_TH - (this.#TA_LOW_TH - t) * this.#TA_LOW_RATIO;
        } else if (t > this.#TA_HIGH_TH) {
            t = this.#TA_HIGH_TH + (t - this.#TA_HIGH_TH) * this.#TA_HIGH_RATIO;
        }
        const subR = this.#ta2r(t) - this.#R_FALL;
        const subTA = this.#r2ta(subR);
        return t - subTA;
    }
    /**
     * 設定された基準TAおよび感度における1TAあたりの目盛の変化量を計算する。
     */
    #calcScaleCoef() {
        const corrGain = this.#taDiffFallBase / this.#taDiffFall(this.#taPosition);
        const r = 1.0 / (1.0 / (this.#sens * this.#R_SENS_BASE) + 1.0 / this.#R_TA20);
        const dTA = this.#TA_BASE - this.#r2ta(r);
        this.#taScaleCoef = ((this.#SCALE_MAX - this.#SCALE_SET) / dTA) * corrGain;
    }
}