const SMA_SIZE = 6;     //100[ms]
const RC_FILTER = 0.82; //0.7-0.95

const AUTOSET_WAIT = 30; //500[ms]
const AUTOSET_LAPS = 2;
const SENS_MIN = 1 / (2 ** 3);
const SENS_RANGE_MAX = 100;
const SENS_RANGE_INIT = 50;

let taPos = 0.0;
let taPC = 0.0;
let outRangeCount = 0;
let filterReset = false;

// メーター
const meter = new SGTMeter();

// Gauge
const GAUGE_MAX_VALUE = 360 / 2.5 * 100;
const gaugeOpts = {
    angle: -0.5,
    lineWidth: 0.07,
    radiusScale: 1.17,
    pointer: {
        length: 0.53,
        strokeWidth: 0.005
    },
    staticZones: [
        { strokeStyle: "#F8F8F8", min: 0, max: GAUGE_MAX_VALUE },  //全体
        { strokeStyle: "#E0E0E0", min: mv2gv(0), max: mv2gv(40) }, //目盛範囲
        { strokeStyle: "#F0F0F0", min: mv2gv(0), max: mv2gv(2) },  //左側範囲
        { strokeStyle: "#F8F8F8", min: mv2gv(12), max: mv2gv(18) },//セット範囲
        { strokeStyle: "#F0F0F0", min: mv2gv(36), max: mv2gv(40) },//テスト範囲
        { strokeStyle: "#0F0F0f", min: mv2gv(0) - 1, max: mv2gv(0) + 1 },   //開始位置
        { strokeStyle: "#0F0F0F", min: mv2gv(15) - 1, max: mv2gv(15) + 1 }, //セット位置
        { strokeStyle: "#0F0F0F", min: mv2gv(26) - 1, max: mv2gv(26) + 1 }, // Fall位置
        { strokeStyle: "#0F0F0F", min: mv2gv(meter.scaleSize) - 1, max: mv2gv(meter.scaleSize) + 1 },//終了位置
    ],
};
const gaugeView = document.getElementById('gaugeView');
const gauge = new Gauge(gaugeView);
gauge.animationSpeed = 1;
gauge.maxValue = GAUGE_MAX_VALUE;
gauge.setOptions(gaugeOpts);
gauge.setMinValue(0);
gauge.set(mv2gv(0));
function mv2gv(mv) { // meter値 → Gauge値
    return GAUGE_MAX_VALUE * (((mv / meter.scaleSize) * (105 / 360)) + (130 / 360));
}

// 感度
const rngSens = document.getElementById("RangeSens");
const lblSens = document.getElementById("LabelSens");
rngSens.max = SENS_RANGE_MAX;

// その他部品
const divCtrl = document.getElementById("DivControl");
const btnConn = document.getElementById("ButtonConnect");
const chkExp = document.getElementById("ChkboxExp");
const lblPos = document.getElementById("LabelTaPos");
const lblTA = document.getElementById("LabelTaPc");
const lblDiff = document.getElementById("LabelTaDiff");
const lblLap = document.getElementById("LabelLaps");

// イベントリスナ登録
document.addEventListener('keydown', (event) => {
    if ((event.shiftKey) || (event.ctrlKey)) {
        taSet();
    } else if (event.key === ' ') {
        chkExp.checked = !chkExp.checked;
    } else if (event.key === 'ArrowUp') {
        rngSens.value = parseInt(rngSens.value) + 10;
        sensChanged();
    } else if (event.key === 'ArrowRight') {
        rngSens.value = parseInt(rngSens.value) + 1;
        sensChanged();
    } else if (event.key === 'ArrowDown') {
        rngSens.value = parseInt(rngSens.value) - 10;
        sensChanged();
    } else if (event.key === 'ArrowLeft') {
        rngSens.value = parseInt(rngSens.value) - 1;
        sensChanged();
    }
});
btnConn.addEventListener('click', () => { start(); });
chkExp.addEventListener('change', () => { chkExp.blur(); });
rngSens.addEventListener('change', () => { sensChanged(); });
rngSens.addEventListener('input', () => { sensChanged(); });
gaugeView.addEventListener('click', () => { taSet(); });

// スクロール抑止
const body = document.querySelector('body');
body.style.top = -window.scrollY + 'px';
body.style.position = 'fixed';

async function start() {
    class LineBreakTransformer {
        constructor() { this.chunks = ""; }
        transform(chunk, controller) {
            this.chunks += chunk;
            const lines = this.chunks.split("\r\n");
            this.chunks = lines.pop();
            lines.forEach((line) => controller.enqueue(line));
        }
        flush(controller) { controller.enqueue(this.chunks); }
    }

    try {
        console.log("connecting...\n");
        const port = await navigator.serial.requestPort();
        await port.open({ baudRate: 15200 });
        console.log("connected\n");
        chkExp.checked = false;
        rngSens.value = SENS_RANGE_INIT;
        taPC = 1.0;
        taSet();
        sensChanged();
        btnConn.style.display = "none";
        divCtrl.style.display = "block";
        let hist = Array(SMA_SIZE).fill(0);
        let meterValue = 0;

        while (port.readable) {
            const textDecoder = new TextDecoderStream();
            const readableStreamClosed = port.readable.pipeTo(textDecoder.writable);
            const reader = textDecoder.readable.pipeThrough(new TransformStream(new LineBreakTransformer())).getReader();

            try {
                while (true) {
                    const { value, done } = await reader.read();
                    if (done) {
                        console.log("done\n");
                        break;
                    }

                    hist = hist.slice(1);
                    hist[SMA_SIZE - 1] = parseFloat(value);
                    let sum = 0;
                    for (let n = 0; n < SMA_SIZE; ++n) {
                        sum += hist[n];
                    }
                    taPC = sum / SMA_SIZE;
                    meter.ta = taPC;
                    lblTA.textContent = taPC.toFixed(3);

                    let taDiff = (taPC - taPos);
                    if ((-0.0001 < taDiff) && (taDiff < 0.0001)) {
                        taDiff = 0;
                    }
                    lblDiff.textContent = ((taDiff >= 0) ? "+" : " ") + taDiff.toFixed(4);

                    if (filterReset) {
                        meterValue = meter.value;
                        filterReset = false;
                    } else {
                        meterValue = RC_FILTER * meterValue + (1 - RC_FILTER) * meter.value;
                    }
                    if (!chkExp.checked) {
                        if (meterValue >= meter.scaleSize) {
                            meterValue = meter.scaleSize;
                            autoSet();
                        } else if (meterValue < 0) {
                            meterValue = 0;
                            autoSet();
                        } else {
                            outRangeCount = 0;
                        }
                    }

                    const gaugeValue = Math.floor(mv2gv(meterValue));
                    const r = -Math.floor(gaugeValue / GAUGE_MAX_VALUE);
                    if (Math.abs(r) >= AUTOSET_LAPS) {
                        taSet();
                    }
                    lblLap.textContent = r;
                    gauge.set(gaugeValue + (r * GAUGE_MAX_VALUE));
                }
            } catch (error) {
                console.log("stream broken\n");
                console.log(error);
            } finally {
                reader.releaseLock();
                await port.close();
                console.log("disconnected\n");
            }
            btnConn.style.display = "inline";
            divCtrl.style.display = "none";
            gauge.set(mv2gv(0));
        }
    } catch (error) {
        console.log("connection failed");
        console.log(error);
    }
}

function taSet() {
    taPos = taPC;
    meter.position = taPos;
    lblPos.textContent = taPos.toFixed(3);
    filterReset = true;
}
function autoSet() {
    ++outRangeCount;
    if (outRangeCount >= AUTOSET_WAIT) {
        taSet();
        outRangeCount = 0;
    }
}
function sensChanged() {
    meter.sens = SENS_MIN * 2 ** (rngSens.value / 10);
    lblSens.textContent = meter.sens.toFixed(2);
    rngSens.blur();
    filterReset = true;
}