/*
 * Copyright (c) 2025 ShinGeTsu Meter.
 * SPDX-License-Identifier: BSD-3-Clause
 */
#include <stdio.h>

#include "hardware/adc.h"
#include "hardware/gpio.h"
#include "pico/stdlib.h"

#define LED_PIN      25
#define VSYS_ADC_PIN 28
#define VSYS_ADC_NO  2
#define TA_MAX       6.5f                   // TA最大値
#define TA_MIN       0.9412f                // TA最小値
#define TA_REG       21250.0f               // TA計算用の抵抗値
#define CONST_V      3.3f                   // ブリッジの定電圧値
#define CONST_R      22000.0f               // ブリッジの固定抵抗
#define SPS          60                     // 1秒間のサンプリング数
#define SAMPLING_uS  (1000 * 1000 / SPS)    // 連続サンプリング時間

static bool timeExceeded = false;    // タイマー用

/** タイマー割り込みハンドラ */
static int64_t alarmHandler(alarm_id_t id, void *user_data) {
    timeExceeded = true;
    return 0;
}

/** メイン */
int main() {
    sleep_ms(100);
    gpio_init(LED_PIN);
    gpio_set_dir(LED_PIN, GPIO_OUT);
    gpio_put(LED_PIN, true);
    adc_init();
    adc_gpio_init(VSYS_ADC_PIN);
    adc_select_input(VSYS_ADC_NO);
    stdio_init_all();
    sleep_ms(200);
    gpio_put(LED_PIN, false);

    uint16_t i = 0;
    bool     led = true;
    while (true) {
        float    sum = 0.0;
        uint16_t count = 0;
        timeExceeded = false;
        add_alarm_in_us(SAMPLING_uS, alarmHandler, NULL, false);
        while (timeExceeded == false) {
            sum += (adc_read() * (CONST_V / 4096.0f));
            ++count;
        }
        float v = sum / count;
        float r = v * CONST_R / (CONST_V - v);                       // 抵抗値計算
        float ta = r / (r + TA_REG) * (TA_MAX - TA_MIN) + TA_MIN;    // TA値計算
        printf("%9.7f\n", ta);                                       // 出力
        if (i >= SPS) {
            gpio_put(LED_PIN, led);
            led = !led;
            i = 0;
        }
        i++;
    }
    return 0;
}