/*
 * Copyright (c) 2025 ShinGeTsu Meter.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */
#include <math.h>
#include <stdio.h>
#include <stdlib.h>

#include "hardware/adc.h"
#include "hardware/gpio.h"
#include "hardware/watchdog.h"
#include "pico/bootrom.h"
#include "pico/stdlib.h"
#include "pico/time.h"

#include "device.h"
#include "meter.h"

// GPIO
static const uint GPIO_PIN[] = {
    BTN1, BTN2, BTN3, BTN4, BTN5, BTN6, BTN7, BTN8, BTN9,    // ボタン
    SE_RE1, SE_RE2, TA_RE1, TA_RE2                           // ロータリーエンコーダ
};

// TA
static float pcV = 0.0f;              // 入力電圧
static float pcTA = TA_MAX;           // PCのTA値
static float setTA = TA_MIN;          // 基準TA値
static float taList[TA_HIST_SIZE];    // PCのTAの履歴
static float taRange = 0.0f;          // 履歴内のTA変動範囲

// TAカウンター
static float  taCounter = 0.0f;                // TAカウンター
static float  counterList[COUNT_HIST_SIZE];    // TAカウンター履歴
static size_t counterListIndex = 0;            // TAカウンター履歴のインデックス

// 感度
static float sens[2] = {SENS_NORM_INITIAL, SENS_LOW_INITIAL};    // 感度 [0]:通常モード、[1]:低感度モード
typedef enum {
    SENS_NORM = 0,
    SENS_LOW = 1
} sensIndex_e;
static sensIndex_e sensIndex = SENS_NORM;    // 感度配列のインデックス
static float       taDiffFallBase = 1.0f;    // 補正倍率を1とするTA値において、針がFALL目盛まで移動するTAの減少量
static float       corrGain = 1.0f;          // 感度補正の倍率

// メーター
static float meterValue = 0.0f;     // メーターの目盛値
static float taScaleCoef = 0.0f;    // 1TAあたりのメーター目盛変化量
static float dacPos = 0;            // DACの値

// イベントフラグ
static bool setEvt = false;       // セット実行
static bool sensEvt = false;      // 感度操作
static bool updateEvt = false;    // LCD表示更新

// 状態管理
static bool autoSet = false;         // 自動セット有効
static bool counting = false;        // TAカウント有効
static bool pause = false;           // TAカウントを中断
static bool powerView = false;       // 電源表示モード
static bool sensCorrect = true;      // 感度補正モード
static bool standby = true;          // 節電中
static bool usbConnected = false;    // USB接続フラグ
static bool timeExceeded = false;    // タイマー用
static bool outOfRange = false;      // 針が範囲外ならtrue

static int64_t autoSetWaitUs = AUTO_SET_WAIT_uS;    // 自動セットの待ち時間
static int16_t samplingCount = 0;                   // SAMPLING_uS間のサンプリング数

/**
 * @brief 電圧からTAを計算する。
 */
static inline float v2ta(float v) {
    return ((v / CONST_V) * (TA_MAX - TA_MIN)) + TA_MIN;
}

/**
 * @brief TAから抵抗値を計算する。
 */
static inline float ta2r(float ta) {
    return CONST_R * (ta - TA_MIN) / (TA_MAX - ta);
}

/**
 * @brief 抵抗値からTAを計算する。
 */
static inline float r2ta(float r) {
    return r / (r + CONST_R) * (TA_MAX - TA_MIN) + TA_MIN;
}

/**
 * @brief 引数のTA値にてPCの抵抗値が CORRECT_R_FALL 下がった際のTAの変化量を計算する。
 *
 * TAがCORRECT_TA_LOW_THからCORRECT_TA_HIGH_THを外れる場合は
 * 感度補正を穏やかにするために換算したTA値にて計算する。
 */
static inline float taDiffFall(float ta) {
    if (ta < CORRECT_TA_LOW_TH) {
        ta = CORRECT_TA_LOW_TH - (CORRECT_TA_LOW_TH - ta) * CORRECT_TA_LOW_RATIO;
    } else if (ta > CORRECT_TA_HIGH_TH) {
        ta = CORRECT_TA_HIGH_TH + (ta - CORRECT_TA_HIGH_TH) * CORRECT_TA_HIGH_RATIO;
    }
    float subR = ta2r(ta) - CORRECT_R_FALL;
    float subTA = r2ta(subR);
    return ta - subTA;
}

/**
 * @brief 現在の感度および基準TA値における1TAあたりのメーター目盛の変化量を計算する。
 *
 * 感度の定義：
 *   ・5kΩ(TA2.0)の抵抗を接続し針をセット位置に設定した状態で、100kΩの抵抗を
 * 　　並列接続した際に、針がメモリの右端まで移動する増幅率が感度32となる。
 *   ・同様に50kΩを並列接続した際に、針がメモリの右端まで移動する増幅率が感度16となる。
 * ⇒ TAが2.0で感度がSのとき、[S × 3.125kΩ]の抵抗を並列接続すると、針がメモリの右端を指す。
 *
 * 感度が補正されていない場合、基準TA値によらずTAの変化量と針の振れ幅は比例する。
 */
static void calcScaleCoef() {
    float r = 1.0f / (1.0f / (sens[sensIndex] * SENS_R_BASE) + 1.0f / SENS_R_TA20);    // TAが2.0の時に針が目盛の右端となる抵抗値
    float dTA = CORRECT_TA_BASE - r2ta(r);                                             // 針が目盛の右端となるTAの差分
    taScaleCoef = (METER_SCALE_MAX - METER_SCALE_SET) / dTA;                           // 1TAあたりの目盛変化量

    if (sensCorrect) {    // 感度補正有効
        corrGain = taDiffFallBase / taDiffFall(setTA);
        taScaleCoef *= corrGain;
    } else {
        corrGain = 1.0f;
    }
}

/**
 * @brief TAアクションをカウントする
 */
static void countTA() {
    if (!pause && counting) {
        float dTA = (setTA - pcTA);
        if (dTA > 0) {
            taCounter += dTA;
            if (taCounter >= 100.0f) taCounter = taCounter - 100.0f;
            ++counterListIndex;
            counterListIndex = counterListIndex % COUNT_HIST_SIZE;
            counterList[counterListIndex] = taCounter;
        }
    }
}

/**
 * @brief TAカウントを履歴から復元する。
 */
static void undoCounter(int num) {
    counterListIndex = (counterListIndex + COUNT_HIST_SIZE + num) % COUNT_HIST_SIZE;
    taCounter = counterList[(counterListIndex + COUNT_HIST_SIZE) % COUNT_HIST_SIZE];
}

/**
 * @brief PCのTA履歴を保存する。履歴からTAの変動範囲を集計する。
 */
static void addTaList(float ta) {
    static size_t listIndex = 0;

    taList[listIndex] = ta;
    float maxValue = 0.0f;
    float minValue = 7.0f;
    for (int i = 0; i < TA_HIST_SIZE; i++) {
        if (maxValue < taList[i]) maxValue = taList[i];
        if (minValue > taList[i]) minValue = taList[i];
    }
    taRange = fabs(maxValue - minValue);

    listIndex = (listIndex + 1) % TA_HIST_SIZE;
}

/**
 * @brief PCのTA履歴ををクリアする。
 *
 */
static void clearTaHist() {
    taRange = 0.0f;
    for (int i = 0; i < TA_HIST_SIZE; i++) {
        taList[i] = pcTA;
    }
}

/**
 * @brief タイマー割り込みハンドラ
 */
static int64_t alarmHandler(alarm_id_t id, void *user_data) {
    timeExceeded = true;
    return 0;
}

/**
 * @brief ボタンイベントを管理し実行可否を判定する。
 *
 * 　前回の実行後 EVENT_WAIT_uS 以上経過している場合trueを返す
 */
static bool checkBtnAction(uint target, uint gpio, uint32_t event) {
    static uint64_t nextAction[29];    // gpio:0-28
    bool            result = false;
    if (target == gpio) {
        uint64_t nowUs = time_us_64();
        result = ((nowUs > nextAction[target]) && (event == GPIO_IRQ_EDGE_FALL));
        nextAction[target] = nowUs + EVENT_WAIT_uS;
    }
    return result;
}

/**
 * @brief GPIO割り込みハンドラ
 */
static void irqHandler(uint gpio, uint32_t events) {
    static const int8_t RE_DIR[] = ROTARY_DIRECTION;
    // ロータリーエンコーダー(感度)
    if (gpio == SE_RE1 || gpio == SE_RE2) {
        static uint8_t re1;
        static uint8_t re2;
        static size_t  reIndex = 0;

        uint8_t state = (events == GPIO_IRQ_EDGE_RISE) ? 1 : 0;
        if (gpio == SE_RE1) re1 = state;
        if (gpio == SE_RE2) re2 = state;
        reIndex = ((reIndex << 2) + (re1 << 1) + re2) & 0xF;

        int8_t dir = RE_DIR[reIndex];
        if ((dir > 0) && (sens[sensIndex] < SENS_MAX)) {
            sens[sensIndex] = sens[sensIndex] * SENS_RE_RATIO;
            sensEvt = true;
        } else if ((dir < 0) && (SENS_MIN < sens[sensIndex])) {
            sens[sensIndex] = sens[sensIndex] / SENS_RE_RATIO;
            sensEvt = true;
        }
    }

    // ロータリーエンコーダー(TA)
    if (gpio == TA_RE1 || gpio == TA_RE2) {
        static uint8_t re1;
        static uint8_t re2;
        static size_t  reIndex = 0;
        static int8_t  dir = 0;

        uint8_t state = (events == GPIO_IRQ_EDGE_RISE) ? 1 : 0;
        if (gpio == TA_RE1) re1 = state;
        if (gpio == TA_RE2) re2 = state;
        reIndex = ((reIndex << 2) + (re1 << 1) + re2) & 0xF;

        if (dir != RE_DIR[reIndex]) {    // 反転時の初回イベントを無視する
            dir = RE_DIR[reIndex];
        } else {
            if ((dir > 0) && (setTA < TA_MAX)) {
                if (meterValue < 0.0f) {
                    setTA = pcTA;
                    calcScaleCoef();
                    setTA = pcTA - (METER_SCALE_SET + 4) / taScaleCoef;
                    while (true) {
                        calcScaleCoef();
                        meterValue = (setTA - pcTA) * taScaleCoef + METER_SCALE_SET;
                        if (meterValue < 0.0f) {
                            setTA += (1 / taScaleCoef);
                        } else {
                            break;
                        }
                    }
                } else {
                    setTA += (TA_RE_STEP / taScaleCoef);
                    calcScaleCoef();
                    meterValue = (setTA - pcTA) * taScaleCoef + METER_SCALE_SET;
                }
            } else if ((dir < 0) && (setTA > TA_MIN)) {
                float tTA = setTA;
                if (meterValue > METER_SCALE_SIZE) {
                    setTA = pcTA;
                    calcScaleCoef();
                    setTA = pcTA + (METER_SCALE_SIZE - METER_SCALE_SET + 4) / taScaleCoef;
                    while (true) {
                        calcScaleCoef();
                        meterValue = (setTA - pcTA) * taScaleCoef + METER_SCALE_SET;
                        if (meterValue > METER_SCALE_SIZE) {
                            setTA -= (1 / taScaleCoef);
                        } else {
                            break;
                        }
                    }
                } else {
                    setTA -= (TA_RE_STEP / taScaleCoef);
                    calcScaleCoef();
                    meterValue = (setTA - pcTA) * taScaleCoef + METER_SCALE_SET;
                }
                if (!pause && counting) {
                    taCounter += (tTA - setTA);
                    counterList[counterListIndex] = taCounter;
                }
            }
        }
    }

    // セットボタン
    else if (checkBtnAction(BUTTON_SET, gpio, events)) {
        setEvt = true;
        pause = false;
    }

    // セットボタンその２
    else if (checkBtnAction(BUTTON_SET2, gpio, events)) {
        setEvt = true;
        pause = false;
    }

    // 自動モード切替ボタン
    else if (checkBtnAction(BUTTON_AUTO, gpio, events)) {
        autoSet = !autoSet;
        updateEvt = true;
    }

    // 電源表示ボタン
    else if (gpio == BUTTON_POWER) {
        updateEvt = true;
    }

    // 感度切替ボタン
    else if (checkBtnAction(BUTTON_SENS, gpio, events)) {
        if (!gpio_get(BUTTON_POWER)) {
            sensCorrect = !sensCorrect;
        } else {
            if (sensIndex == SENS_NORM) {
                sensIndex = SENS_LOW;
            } else {
                sensIndex = SENS_NORM;
            }
        }
        calcScaleCoef();
        updateEvt = true;
    }

    // カウントモード切替ボタン
    else if (checkBtnAction(BUTTON_COUNT, gpio, events)) {
        if (!counting) {
            pause = false;
            counting = true;
        } else {
            counting = false;
        }
        updateEvt = true;
    }

    // クリアボタン
    else if (checkBtnAction(BUTTON_CLEAR, gpio, events)) {
        if (!gpio_get(BUTTON_POWER)) {
            autoSetWaitUs = 0;
        } else {
            undoCounter(1);
            taCounter = 0.0f;
            counterList[counterListIndex] = taCounter;
            pause = false;
        }
        updateEvt = true;
    }

    // 戻るボタン
    else if (checkBtnAction(BUTTON_BACK, gpio, events)) {
        if (!gpio_get(BUTTON_POWER)) {
            autoSetWaitUs -= AUTO_SET_ADJ_STEP;
            if (autoSetWaitUs < 0) {
                autoSetWaitUs = 0;
            }
        } else {
            undoCounter(-1);
        }
        updateEvt = true;
    }

    // 進むボタン
    else if (checkBtnAction(BUTTON_FORWARD, gpio, events)) {
        if (!gpio_get(BUTTON_POWER)) {
            autoSetWaitUs += AUTO_SET_ADJ_STEP;
            if (autoSetWaitUs > AUTO_SET_WAIT_MAX_uS) {
                autoSetWaitUs = AUTO_SET_WAIT_MAX_uS;
            }
        } else {
            undoCounter(1);
        }
        updateEvt = true;
    }
}

/**
 * @brief GPIO初期化
 */
static void initIO() {
    // LED
    gpio_init(LED_PIN);
    gpio_set_dir(LED_PIN, GPIO_OUT);

    // USB接続チェック用
    gpio_init(VBUS_SENSE_PIN);
    gpio_set_dir(VBUS_SENSE_PIN, GPIO_IN);

    // 電源チェック用
    adc_init();
    adc_gpio_init(VSYS_ADC_PIN);

    // GPIO
    uint size = sizeof(GPIO_PIN) / sizeof(uint);
    for (int i = 0; i < size; ++i) {
        gpio_init(GPIO_PIN[i]);
        gpio_set_dir(GPIO_PIN[i], GPIO_IN);
        gpio_pull_up(GPIO_PIN[i]);
        gpio_set_irq_enabled_with_callback(GPIO_PIN[i], GPIO_IRQ_EDGE_FALL, true, irqHandler);
        gpio_set_irq_enabled_with_callback(GPIO_PIN[i], GPIO_IRQ_EDGE_RISE, true, irqHandler);
    }
}

/**
 * @brief LCDを更新する。
 */
static void update() {
    char       *statusLabel;
    static char message[30];
    static char sensLabel[5];
    static char divTaS[7];

    if (!powerView && gpio_get(BUTTON_POWER)) {
        if (setEvt) {
            statusLabel = "SET";
        } else {
            char tlabel[5];
            tlabel[0] = (sensIndex == SENS_LOW) ? '*' : ' ';
            tlabel[1] = (autoSet) ? 'a' : ' ';
            tlabel[2] = (counting) ? 'c' : ' ';
            tlabel[3] = (counting && pause) ? '!' : ' ';
            tlabel[4] = '\0';
            statusLabel = tlabel;
        }
        if (sens[sensIndex] < 0.1f) {
            sprintf(sensLabel, ".%03d", (int16_t)(sens[sensIndex] * 1000));
        } else if (sens[sensIndex] < 1.0f) {
            sprintf(sensLabel, "%4.2f", sens[sensIndex]);
        } else if (sens[sensIndex] < 10.0f) {
            sprintf(sensLabel, "%4.1f", sens[sensIndex]);
        } else if (sens[sensIndex] < 100.0f) {
            sprintf(sensLabel, "%4.0f", sens[sensIndex]);
        } else {
            sprintf(sensLabel, " %3d", (int16_t)(sens[sensIndex]));
        }
        if (standby) {
            sprintf(divTaS, "------");
        } else {
            sprintf(divTaS, "%+6.3f", pcTA - setTA);
        }
        sprintf(message, "%4.2f %s %s", pcTA, divTaS, statusLabel);
        lcdPrint(1, message);
        sprintf(message, "%4.2f %s %6.3f", setTA, sensLabel, taCounter);
        lcdPrint(2, message);
    } else {
        // 電源電圧計測
        adc_select_input(VSYS_ADC_NO);
        float vsys = adc_read() * (3.3f * 3.0f / 4096.0f);
        if (usbConnected) {
            statusLabel = "USB :";
        } else {
            statusLabel = "BATT:";
            vsys += 0.18f;    // ショットキーバリアダイオード分
        }
        sprintf(message, "%s%4.2fv   %s", statusLabel, vsys, sensCorrect ? "Reg" : " TA");
        lcdPrint(1, message);
        sprintf(message, "auto set:%3.1fsec ", (float)(autoSetWaitUs / 1000.0f / 1000.0f));
        lcdPrint(2, message);
    }
}

/**
 * @brief 状態を確認する。
 */
static void checkState() {
    static int64_t lastUs = 0;
    if (autoSet && !standby) {
        int64_t nowUs = time_us_64();
        if ((taRange > TA_PAUSE_RANGE)) {
            pause = true;
        }
        if (outOfRange) {
            if (nowUs >= (lastUs + autoSetWaitUs)) {
                setEvt = true;
                lastUs = nowUs;
            }
        } else {
            lastUs = nowUs;
        }
    }
}

/**
 * @brief SAMPLING_uS間 メーターを駆動する。
 */
static void driveNeedle() {
    samplingCount = 0;
    timeExceeded = false;
    outOfRange = true;

    if (standby) {
        adcStart();
        standby = false;
    }

    float scaleSum = 0.0f;
    float pcTASum = 0.0f;
    add_alarm_in_us(SAMPLING_uS, alarmHandler, NULL, false);
    while (!timeExceeded) {
        ++samplingCount;
        pcV = adcRead();
        pcTA = v2ta(pcV);
        pcTASum += pcTA;
        if (pcTA < PC_TA_MAX) {
            float scale = (setTA - pcTA) * taScaleCoef + METER_SCALE_SET;
            scaleSum += scale;
            if (scale > 0.0f) {
                dacPos = scale * ((DAC_MAX_VALUE) / METER_SCALE_SIZE);
            } else {
                dacPos = 0.0f;
            }
            dacPos = meterDacSet(dacPos);
        } else {
            adcStop();    // 節電
            dacPos = meterDacSet(0.0f);
            standby = true;
            pause = true;
            pcTASum = 6.5f;
            scaleSum = 0.0f;
            samplingCount = 1;
            break;
        }
        sleep_us(ADC_WAIT_US);
    }

    pcTA = pcTASum / samplingCount;
    meterValue = scaleSum / samplingCount;
    if (meterValue >= 999.99f) meterValue = 999.99f;
    if (meterValue <= -999.99f) meterValue = -999.99f;

    if ((0.0f < meterValue) && (meterValue <= METER_SCALE_SIZE)) {
        outOfRange = false;
    }
}

/**
 * @brief メイン
 */
int main() {
    sleep_ms(100);
    initDevice();
    initIO();
    sleep_ms(100);

    usbConnected = gpio_get(VBUS_SENSE_PIN);
    bool debugout = (!gpio_get(BUTTON_POWER));
    if (debugout) {
        stdio_init_all();
    }

    // 設定モードを表示
    powerView = true;
    update();
    uint64_t waitUs = time_us_64() + POWER_VIEW_WAIT_uS;
    while (powerView) {
        if ((time_us_64() > waitUs) || !gpio_get(BUTTON_POWER)) {
            powerView = false;
            break;
        } else if (!gpio_get(BUTTON_AUTO) && !gpio_get(BUTTON_SENS) && usbConnected) {
            lcdPrint(1, "Device Firmware ");
            lcdPrint(2, "Update          ");
            reset_usb_boot(0, 0);
        }
        sleep_ms(100);
    }

    autoSet = false;
    setEvt = false;
    sensEvt = false;
    updateEvt = true;
    sensCorrect = true;
    sens[SENS_NORM] = SENS_NORM_INITIAL;
    sens[SENS_LOW] = SENS_LOW_INITIAL;
    setTA = TA_MIN;
    taDiffFallBase = taDiffFall(CORRECT_TA_BASE);
    calcScaleCoef();

    uint64_t netxUpdateUs = 0;
    while (true) {
        checkState();
        if (debugout) {
            char   *label = " ";
            int32_t sps = samplingCount * (1000000 / SAMPLING_uS);
            if (setEvt) label = "SET";
            printf("SPS:%-03d V:%-8.3f pc:%-07.5f set:%07.5f sens:%08.4f gain:%07.3f div:%+08.5f meter:%+07.2f dac:%04d %s\n",
                   sps, pcV * 1000.0f, pcTA, setTA, sens[sensIndex], corrGain, pcTA - setTA, meterValue, (int16_t)dacPos, label);
        }
        // 感度変更イベント
        if (sensEvt) {
            calcScaleCoef();
            update();
            netxUpdateUs = time_us_64() + UPDATE_uS;
            sensEvt = false;
        }
        // セットイベント
        else if (setEvt) {
            gpio_put(LED_PIN, true);
            update();
            countTA();
            setTA = pcTA;
            calcScaleCoef();
            clearTaHist();
            netxUpdateUs = time_us_64() + UPDATE_uS;    // UPDATE_uSの間 "SET"を表示する
            outOfRange = false;
            setEvt = false;
            gpio_put(LED_PIN, false);
        }
        // 表示更新イベント
        else if (updateEvt) {
            update();
            netxUpdateUs = time_us_64() + UPDATE_uS;
            updateEvt = false;
        }
        // イベント外
        else {
            if (standby) {
                sleep_us(SAMPLING_uS);
            } else {
                driveNeedle();
                addTaList(pcTA);
                if (standby) {
                    update();
                    netxUpdateUs = time_us_64() + UPDATE_uS;
                }
            }
            // UPDATE_uS以上経過していたらLCDを更新
            uint64_t nowUs = time_us_64();
            if (nowUs >= netxUpdateUs) {
                if (standby) {
                    driveNeedle();
                    addTaList(pcTA);
                }
                update();
                netxUpdateUs = nowUs + UPDATE_uS;
            }
        }
    }
    return 0;
}