/*
 * Copyright (c) 2025 ShinGeTsu Meter.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */
#include <math.h>

#include "hardware/gpio.h"
#include "hardware/i2c.h"

#include "aqm1602.h"
#include "mcp356xR.h"
#include "mcp4726.h"

#include "device.h"

static float corr = 1.0f;    // ゲイン誤差(?)校正用

/**
 * @brief デバイスを初期化
 */
void initDevice() {
    // i2c0 : メーター用DAC
    i2c_init(i2c0, I2C0_SPEED_kHz * 1000);
    gpio_set_function(I2C0_SDA_PIN, GPIO_FUNC_I2C);
    gpio_set_function(I2C0_SCL_PIN, GPIO_FUNC_I2C);
    gpio_pull_up(I2C0_SDA_PIN);
    gpio_pull_up(I2C0_SCL_PIN);
    MCP4726_Init(i2c0, ADDR_MCP4726);
    MCP4726_Set(ADDR_MCP4726, 0);

    // i2c1 : LCD
    i2c_init(i2c1, I2C1_SPEED_kHz * 1000);
    gpio_set_function(I2C1_SDA_PIN, GPIO_FUNC_I2C);
    gpio_set_function(I2C1_SCL_PIN, GPIO_FUNC_I2C);
    gpio_pull_up(I2C1_SDA_PIN);
    gpio_pull_up(I2C1_SCL_PIN);
    LCD_Init(i2c1, ADDR_LCD);

    // spi0 : ADC
    spi_init(spi0, SPI0_SPEED_kHz * 1000);
    gpio_set_function(SPI0_SCK, GPIO_FUNC_SPI);
    gpio_set_function(SPI0_MOSI, GPIO_FUNC_SPI);
    gpio_set_function(SPI0_MISO, GPIO_FUNC_SPI);
    gpio_init(SPI0_CS);
    gpio_set_dir(SPI0_CS, GPIO_OUT);
    gpio_put(SPI0_CS, 1);
    MCP356xR_Init(spi0, SPI0_CS);
}

void adcStart() {
    MCP356xR_Select(MCP356xR_CH_REF_P, MCP356xR_CH_REF_N);
    MCP356xR_Start();
    sleep_us(ADC_WAIT_US);
    float v = MCP356xR_Read(true);
    corr = MCP356xR_REFV / v;
    MCP356xR_Select(MCP356xR_CH_0, MCP356xR_CH_REF_N);
    MCP356xR_Start();
    sleep_us(ADC_WAIT_US);
}

void adcStop() {
    MCP356xR_Stop();
}

float adcRead() {
    return MCP356xR_Read(true) * corr;
}

int16_t meterDacSet(float value) {
    if (value >= INT16_MAX) value = INT16_MAX;
    return MCP4726_Set(ADDR_MCP4726, (int16_t)value);
}

/**
 * @brief LCDへメッセージを表示する
 */
void lcdPrint(uint8_t row, char *message) {
    LCD_SelectRow(row);
    LCD_Write(message);
}
