/*
 * Copyright (c) 2025 ShinGeTsu Meter.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */
#include "hardware/gpio.h"
#include "hardware/spi.h"

#include "mcp356xR.h"

static spi_inst_t *spi;
static uint        cspin;

static void spiSelect(bool b) {
    asm volatile("nop \n nop \n nop");
    gpio_put(cspin, !b);
    asm volatile("nop \n nop \n nop");
}

static void spiRead(const uint8_t *src, uint8_t *dst, size_t len) {
    spiSelect(true);
    spi_write_read_blocking(spi, src, dst, len);
    spiSelect(false);
}

static void spiWrite(uint8_t *pData, uint16_t size) {
    spiSelect(true);
    spi_write_blocking(spi, pData, size);
    spiSelect(false);
}

static void fastCommand(uint8_t cmdBit) {
    uint8_t cmd = MCP356xR_DEVADDR | cmdBit | MCP356xR_FAST_CMD;
    spiWrite(&cmd, 1);
}

void MCP356xR_Init(spi_inst_t *s, uint pin) {
    spi = s;
    cspin = pin;
    uint8_t cmd[2];

    fastCommand(MCP356xR_CMD_RESET);

    uint8_t cmdBase = MCP356xR_DEVADDR | MCP356xR_WRITE_INC;

    cmd[0] = cmdBase | MCP356xR_REG_CONFIG0;
    cmd[1] = MCP356xR_VAL_CONFIG0;
    spiWrite(cmd, 2);

    cmd[0] = cmdBase | MCP356xR_REG_CONFIG1;
    cmd[1] = MCP356xR_VAL_CONFIG1;
    spiWrite(cmd, 2);

    cmd[0] = cmdBase | MCP356xR_REG_CONFIG2;
    cmd[1] = MCP356xR_VAL_CONFIG2;
    spiWrite(cmd, 2);

    cmd[0] = cmdBase | MCP356xR_REG_CONFIG3;
    cmd[1] = MCP356xR_VAL_CONFIG3;
    spiWrite(cmd, 2);

    cmd[0] = cmdBase | MCP356xR_REG_IRQ;
    cmd[1] = MCP356xR_VAL_IRQ;
    spiWrite(cmd, 2);
}

void MCP356xR_Select(uint8_t ch_p, uint8_t ch_n) {
    uint8_t cmd[4] = {0, 0, 0, 0};
    cmd[0] = MCP356xR_DEVADDR | MCP356xR_REG_MUX | MCP356xR_WRITE_INC;
    cmd[1] = (ch_p << 4) | ch_n;
    spiWrite(cmd, 2);
}

float MCP356xR_Read(bool wait) {
    uint8_t val[5] = {0, 0, 0, 0, 0};
    uint8_t cmd[5] = {0, 0, 0, 0, 0};

    // 変換待ち
    if (wait) {
        cmd[0] = MCP356xR_DEVADDR | MCP356xR_REG_IRQ | MCP356xR_READ_STATIC;
        while (true) {
            spiRead(cmd, val, 2);
            if ((val[1] & MCP356xR_IRQ_DR_STATUS) == 0) break;
        }
    }

    // 読み出し
    cmd[0] = MCP356xR_DEVADDR | MCP356xR_REG_ADCDATA | MCP356xR_READ_STATIC;
    spiRead(cmd, val, 5);
    uint32_t adcVal = (val[1] << 24) | (val[2] << 16) | (val[3] << 8) | val[4];
    return (float)(MCP356xR_REFV * ((int32_t)adcVal) / MCP356xR_ADCMAX);
}

void MCP356xR_Start() {
    fastCommand(MCP356xR_CMD_START);
}

void MCP356xR_Stop() {
    fastCommand(MCP356xR_CMD_ADC_SHUTDOWN);
}