/*
 * Copyright (c) 2025 ShinGeTsu Meter.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */
#include <stdio.h>

#include "hardware/gpio.h"
#include "pico/stdlib.h"
#include "pico/time.h"

#include "mcp356xR.h"

// IO
#define LED_PIN        25
#define SPI0_SCK       18
#define SPI0_MOSI      19
#define SPI0_MISO      16
#define SPI0_CS        17
#define SPI0_SPEED_kHz 40

// TA計算用
#define TA_MAX  6.5f        // TA最大値
#define TA_MIN  0.9412f     // TA最小値
#define TA_REG  21250.0f    // TA計算用の抵抗値
#define CONST_V 2.4f        // ブリッジの定電圧値 ・・・ MCP356xRの内部参照電圧
#define CONST_R 22000.0f    // ブリッジの固定抵抗

// ADC
#define SPS 60    // 1秒間のサンプリング数

static float corr = 1.0f;    // ゲイン誤差(?)校正用

/**
 * @brief ADC初期化
 */
static void adcInit() {
    spi_init(spi0, SPI0_SPEED_kHz * 1000);
    gpio_set_function(SPI0_SCK, GPIO_FUNC_SPI);
    gpio_set_function(SPI0_MOSI, GPIO_FUNC_SPI);
    gpio_set_function(SPI0_MISO, GPIO_FUNC_SPI);
    gpio_init(SPI0_CS);
    gpio_set_dir(SPI0_CS, GPIO_OUT);
    gpio_put(SPI0_CS, 1);
    MCP356xR_Init(spi0, SPI0_CS);

    MCP356xR_Select(MCP356xR_CH_REF_P, MCP356xR_CH_REF_N);
    MCP356xR_Start();
    float v = MCP356xR_Read(true);
    corr = MCP356xR_REFV / v;
    MCP356xR_Select(MCP356xR_CH_0, MCP356xR_CH_REF_N);
    MCP356xR_Start();
}

/**
 * @brief メイン
 */
int main() {
    sleep_ms(100);
    gpio_init(LED_PIN);
    gpio_set_dir(LED_PIN, GPIO_OUT);
    gpio_put(LED_PIN, true);
    adcInit();
    stdio_init_all();
    sleep_ms(200);
    gpio_put(LED_PIN, false);

    uint16_t i = 0;
    bool     led = true;
    while (true) {
        float v = MCP356xR_Read(true) * corr;
        float r = v * CONST_R / (CONST_V - v);
        float ta = r / (r + TA_REG) * (TA_MAX - TA_MIN) + TA_MIN;
        printf("%9.7f\n", ta);
        if (i >= SPS) {
            gpio_put(LED_PIN, led);
            led = !led;
            i = 0;
        }
        i++;
    }
    return 0;
}